;=======================================================================
; ASM86 assembly language drivers for Intel's byte-wide FlashFile 
; memory family
;=======================================================================

;=======================================================================
; Copyright Intel Corporation, 1996
; EXAMPLE ASM86 Drivers for Intel's byte-wide FlashFile memory family
; Author: Ken McKee, Intel Corporation
;
; Revision 1.2, September 13, 1996    Added S3/S5 suuport
; Revision 1.1, July 17, 1996         Added SA/L and SC support 
; Revision 1.0, January 1, 1996       Initial release
;
; NOTE:
; The code assumes 32-bit flat model protected mode for simplicity.
; i.e. ES contains 0 and EDI accesses the entire memory space.
;=======================================================================

TEXT    segment byte    public  'CODE'
	assume  cs:TEXT

; Following is the structure by which all parameters are passed.
params STRUCT           
	erase_addr          DD   ?    ; base of block or device
							; to erase.
	write_addr          DD   ?    ; address to write to.
	write_base          DD   ?    ; base address of block
							; written to.
	read_addr           DD   ?    ; address to read from.
	read_base           DD   ?    ; base address of block
							; read from.
	read_id             DD   ?    ; address to read device
							; code from.
	lock_addr           DD   ?    ; base address of block
							; to lock.
	data_addr           DD   ?    ; address of data to program.
	data                DW   ?    ; data byte to program.
params ENDS

;=======================================================================
; Defines
;=======================================================================
;
;=======================================================================
; Error Codes
;=======================================================================
NO_ERROR                 DW      0
VPP_ERROR                DW      1
PROGRAM_ERROR            DW      2
ERASE_ERROR              DW      3
BLOCK_PROTECTION_ERROR   DW      4      ; Only valid for 28F0xxS3/S5/SC
COMMAND_SEQ_ERROR        DW      5

;=======================================================================
; MACRO set_pin
; This macro pushes parameters needed for the set_pin routine, calls 
; set_pin, and then pops those parameters. set_pin is an implementation-
; dependent function which sets a given pin on the standard 28F0xxS3/S5/SC
; pinout HIGH = 1 or LOW = 0. This macro is used by 28F0xxS3/S5/SC procedure 
; calls.
; 
; Data needed at the beginning of this macro: 
; level:        high voltage on/off
; Trashes:      CL
;=======================================================================
set_pin     MACRO    level
	push level                         ; Push logic level of pin
	call near ptr set_pin              ; Call set_pin
	pop  CL
ENDM

;=======================================================================
; PROCEDURE     block_erase
; This procedure erases a 64K byte block on the 28F008SA and 28F0xxS3/S5/SC.
; 
; Param fields needed: 
;               erase_address: offset of base of block to erase
; Output        AL: holds SR information
;=======================================================================
block_erase  proc    near
	mov  EDI, params.erase_addr
	mov  BYTE PTR ES:[EDI],020H        ; Block Erase command
	mov  BYTE PTR ES:[EDI],0D0H        ; Erase Confirm command

	; Note that it is not strictly necessary to write an erase command 
	; to the base of a block any address within the block will do.

WSM_busy:
	mov  AL,ES:[EDI]                   ; Read SR.
	test AL,080H                       ; If SR.7 = 0, test sets ZF.
	
	; Erase may be suspended here to read/program from/to a different 
	; block.
	
	jz   SHORT WSM_busy                ; Loop while ZF is set.
	mov  BYTE PTR ES:[EDI],050H        ; Clear Status Registers command and
								; place into read mode.
	ret                                ; Return to calling routine.
block_erase  endp
 
;=======================================================================
; PROCEDURE     program
; This procedure programs data to the 28F008SA and 28F0xxS3/S5/SC. 
; 
; Param fields needed: 
;         params.data: data word to be written
;         params.write_addr: offset of address to write
; Output: AL: holds SR information
;=======================================================================
program   proc    near
	mov  EDI,params.write_addr
	mov  BYTE PTR ES:[EDI],040H        ; Program command
	mov  BYTE PTR ES:[EDI],params.data ; Write data.

WSM_busy1:
	mov  AL,ES:[EDI]                   ; Read SR
	test AL,080H                       ; If SR.7 = 0, test sets ZF.
	
	; Program operation may be suspended here to read from a different block.
	
	jz   SHORT WSM_busy1               ; Loop while ZF is set.
	mov  BYTE PTR ES:[EDI],050H        ; Clear Status Registers command and
								; place into read mode.
	ret                                ; Return to calling routine.
program   endp
 
;=======================================================================
; PROCEDURE     erase_suspend_to_read
; This procedure suspends an erase operation to do a read. This procedure
; works for the 28F008SA and 28F0xxS3/S5/SC.
; 
; Note: This procedure assumes that an erase operation is underway.
;
; Param fields needed: 
;         params.read_addr: offset of address to read
; Output: AL: holds SR information
;         CL: data read from the address in params.read_addr
;=======================================================================
erase_suspend_to_read   proc    near
	mov  EDI,params.read_addr          ; Set up offset of erase address.
	mov  BYTE PTR ES:[EDI],0B0H        ; Erase Suspend command

WSM_busy2:
	mov  AL,ES:[EDI]                   ; Read SR from any address.
	test AL,080H                       ; If SR.7 = 0, test sets ZF.
	jz   SHORT WSM_busy2               ; Loop while ZF is set.
	mov  BYTE PTR ES:[EDI],0FFH        ; Read Flash command
	mov  CL,ES:[EDI]                   ; Do actual read put result in CL.
	
	; Arbitrary number of reads can be done here.

	mov  BYTE PTR ES:[EDI],070H        ; Read SR command
	mov  AL,ES:[EDI]                   ; Read SR from any address.
	test AL,040H                       ; If SR.6 = 0, indicating that there 
								; is no erase suspended, test sets ZF.
	jz   SHORT continue                ; Jump to continue if ZF is set.
	mov  BYTE PTR ES:[EDI],0D0H        ; Erase Resume command

continue:
	ret                                ; Return to calling routine.
erase_suspend_to_read   endp
 
;=======================================================================
; PROCEDURE     erase_suspend_to_program
; This procedure suspends an erase operation to do a program. This 
; procedure works for the 28F0xxS3/S5/SC only.
; 
; Note: This procedure assumes that an erase operation is underway.
;
; Param fields needed: 
;         params.write_addr: offset of block to erase
;         params.params.data: offset of address to read
; Output: AL: holds SR information
;=======================================================================
erase_suspend_to_program   proc    near
	mov  EDI,params.write_addr         ; Set up offset of erase address.
	mov  BYTE PTR ES:[EDI],0B0H        ; Erase Suspend command

WSM_busy3:
	mov  AL,ES:[EDI]                   ; Read SR from any address.
	test AL,080H                       ; If SR.7 = 0, test sets ZF.
	jz   SHORT WSM_busy3               ; Loop while ZF is set.
	mov  BYTE PTR ES:[EDI],040H        ; Program command
	mov  BYTE PTR ES:[EDI],params.data ; Write data.

WSM_busy4:
	mov  AL,ES:[EDI]                   ; Read SR
	test AL,080H                       ; If SR.7 = 0, test sets ZF.
	jz   SHORT WSM_busy4               ; Loop while ZF is set.
	
	; Arbitrary number of program operations can be done.

	mov  AL,ES:[EDI]                   ; Read SR from any address.
	test AL,004H                       ; If SR.2 = 0, indicating that there 
								; is no erase suspended, test sets ZF.
	jz   SHORT continue1               ; Jump to continue if ZF is set. 
	mov  BYTE PTR ES:[EDI],0D0H        ; Erase Resume command

continue1:
	ret                                ; Return to calling routine.
erase_suspend_to_program   endp

;=======================================================================
; PROCEDURE     program_suspend_to_read
; This procedure suspends a program operation to do a read. This procedure
; works for the 28F0xxS3/S5/SC.
; 
; Note: This procedure assumes that a program operation is underway.
;
; Param fields needed: 
;         params.read_addr: offset of address to read
; Output: AL: holds SR information
;         CL: data read from the address in params.read_addr
;=======================================================================
program_suspend_to_read     proc      near
	mov  EDI,params.read_addr          ; Set up offset of erase address.
	mov  BYTE PTR ES:[EDI],0B0H        ; Erase Suspend command

WSM_busy5:
	mov  AL,ES:[EDI]                   ; Read SR from any address.
	test AL,080H                       ; If SR.7 = 0, test sets ZF.
	jz   SHORT WSM_busy5               ; Loop while ZF is set.
	mov  BYTE PTR ES:[EDI],0FFH        ; Read Flash command
	mov  CL,ES:[EDI]                   ; Do actual read; put result in CL.
	
	; Arbitrary number of reads can be done here.

	mov  BYTE PTR ES:[EDI],070H        ; Read SR command
	mov  AL,ES:[EDI]                   ; Read SR from any address.
	test AL,004H                       ; If SR.2 = 0, indicating that there 
								; is no erase suspended, test sets ZF.
	jz   SHORT continue2               ; Jump to continue if ZF is set.
	mov  BYTE PTR ES:[EDI],0D0H        ; Erase Resume command

continue2:
	ret                                ; Return to calling routine.
program_suspend_to_read     endp

;=======================================================================
; PROCEDURE     set_block_lock_bit
; This procedure sets a block lock-bit. This procedure works for 
; the 28F0xxS3/S5/SC.
;
; Param fields needed: 
;         params.lock_addr: offset of base of block 
; Output: AL: holds SR information
;=======================================================================
block_lock_bit_set     proc    near

	mov  EDI,params.lock_addr          ; Set up offset of address.
	
	; If Master lock-bit is set, RP = Vhh.
	; set_pin 1                        ; Enable high voltage on to RP#.
	
	mov  BYTE PTR ES:[EDI],060H        ; Set Block Lock-Bit command.
	mov  BYTE PTR ES:[EDI],001H        ; Confirmation.
	
WSM_busy6:
	mov  AL,ES:[EDI]                   ; Read SR from any address.
	test AL,080H                       ; If SR.7 = 0, test sets ZF.
	jz   SHORT WSM_busy6               ; Loop while ZF is set
	
	; If Master lock-bit was set return RP to Vih.
	; set_pin 0                        ; Disable high voltage on RP#.
	
	mov  BYTE PTR ES:[EDI],050H        ; Clear Status Registers command and
								; place into read mode.
	ret                                ; Return to calling routine.
set_block_lock_bit      endp
 
;=======================================================================
; PROCEDURE     set_master_lock_bit
; This procedure sets the master lock-bit. This procedure works for 
; the 28F0xxS3/S5/SC.
;
; Param fields needed: 
;         params.lock_addr: offset of base of block
; Output: AL: holds SR information
;=======================================================================
set_master_lock_bit     proc    near

	mov  EDI,params.lock_addr          ; Set up offset of address.
	set_pin 1                          ; Enable high voltage on RP#.
	mov  BYTE PTR ES:[EDI],060H        ; Set Master Lock-Bit command.
	mov  BYTE PTR ES:[EDI],0F1H        ; Confirmation.
	
WSM_busy7:
	mov  AL,ES:[EDI]                   ; Read SR from any address.
	test AL,080H                       ; If SR.7 = 0, test sets ZF.
	jz   SHORT WSM_busy7               ; Loop while ZF is set
	set_pin 0                          ; Disable high voltage on RP#.
	mov  BYTE PTR ES:[EDI],050H        ; Clear Status Registers command and
								; place in read mode.
	ret                                ; Return to calling routine.
set_master_lock_bit      endp

;=======================================================================
; PROCEDURE     clear_block_lock_bits
; This procedure clears all block lock-bits simultaneously. This procedure 
; works for the 28F0xxS3/S5/SC.
;
; Param fields needed: 
;         params.lock_addr: offset of base of 28F0xxS3/S5/SC block to lock
; Output: AL: holds SR information      
;=======================================================================
clear_block_lock_bits      proc    near

	mov  EDI,params.lock_addr          ; Set up offset of address.
	
	; If Master lock-bit is set, RP = Vhh.
	; set_pin 1                        ; Drive high voltage on RP#.
				
	mov  BYTE PTR ES:[EDI],060H        ; Clear Block Lock-Bits command
	mov  BYTE PTR ES:[EDI],0D0H        ; Confirmation 
	
WSM_busy8:
	mov  AL,ES:[EDI]                   ; Read SR from any address.
	test AL,080H                       ; If SR.7 = 0, test sets ZF.
	jz   SHORT WSM_busy8               ; Loop while ZF is set.
	
	; If Master lock-bit was set return RP to Vih.
	; set_pin 0                        ; Drive high voltage on RP#.
	
	mov  BYTE PTR ES:[EDI],050H        ; Clear Status Registers command and
								; place into read mode.
	ret                                ; Return to calling routine.
clear_block_lock_bits      endp

;=======================================================================
; PROCEDURE     read_identifier_codes
; This procedure provides access to device codes
;
; Param fields needed: 
;         params.read_id: offset of base address
; Output: AL: holds SR information      
;=======================================================================
read_identifier_codes      proc    near

	mov  EDI,params.read_id            ; Set up offset of address.
	mov  BYTE PTR ES:[EDI],090H        ; Read Identifier Codes command.
	mov  AL,ES:[EDI]                   ; Device code data.
	ret                                ; Return to calling routine.
read_identifier_codes       endp

;=======================================================================
; PROCEDURE     SR_full_status_check
; This procedure performs a full Status Register check. It is valid for 
; program, block erase, block-bit set and block-bit reset operations.
;
; Note: This procedure assumes the Status Register data reside in the 
; AL. This information is place in AL after the completion of each
; operation. If an error is detected, the previous operation should
; be executed again.
;
; Output:       CL : error code
;=======================================================================
SR_full_status_check      proc    near

vpp_check:                                      
	test AL,008H                       ; If SR.3 = 0, test sets ZF.
	jz   device_protection_check       ; Next check if ZF is set.
	mov  CL,VPP_ERROR                  ; Place error code in CL.
	jmp  continue3                     ; Jump to end of SR check.

device_protection_check:
	test AL,002H                       ; If SR.1 = 0, test sets ZF.
	jz   command_seq_check             ; Next check if ZF is set.
	mov  CL,BLOCK_PROTECTION_ERROR     ; Place error code in CL.
	jmp  continue3                     ; Jump to end of SR check.

command_seq_check:
	test AL,010H                       ; If SR.4 = 0, test sets ZF.
	jz   block_erase_check             ; Next check if ZF is set.
	test AL,020H                       ; If SR.5 = 0, test sets ZF.
	jz   program_check                 ; Next check if ZF is set.
	mov  CL,COMMAND_SEQ_ERROR          ; Place error code in CL.
	jmp  continue3                     ; Jump to end of SR check.

block_erase_check:
	test AL,020H                       ; If SR.5 = 0, test sets ZF.
	jz   no_error_detected             ; No error detected if ZF is set. 
	mov  CL,ERASE_ERROR                ; Place error code in CL.
	jmp  continue3                     ; Jump to end of SR check.

program_check: 
	mov     CL,PROGRAM_ERROR           ; Place error code in CL. 
	jmp     continue3                  ; Jump to end of SR check.

no_error_detected:
	mov     CL,NO_ERROR                ; Place error code in CL.

continue3:
	ret                                ; Return to calling routine.
SR_full_status_check      endp

TEXT     ends

END
